import 'package:dartz/dartz.dart';
import 'package:flutter_polyline_points/flutter_polyline_points.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';
import 'package:razinsoft_rider/data/repositories/base_repository.dart';
import 'package:razinsoft_rider/domain/interfaces/google_api_service_interface.dart';

import '../../core/errors/failure.dart';
import '../models/place_model.dart';
import '../models/waypoint.dart';
import 'interfaces/way_point_repo_interface.dart';

class GoogleAPIRepoImpl extends BaseRepository implements IGoogleAPIRepo {
  final IGoogleApiService _googleApiService;

  GoogleAPIRepoImpl(this._googleApiService);
  @override
  Future<Either<Failure, String>> getAddressFromLatLng(LatLng latLng) async => await safeApiCall(() async {
      final response = await _googleApiService.getAddressFromLatLng(latLng);
      final data = response.data;
      if (data['status'] == 'OK') {
        final List<dynamic> results = data['results'];
        if (results.isNotEmpty) {
          final firstResult = results[0];
          final formattedAddress = firstResult['formatted_address'];
          return formattedAddress;
        }
      }
      return 'Unknown Location';
    });

  @override
  Future<List<LatLng>> getWayPoints() {
    throw UnimplementedError();
  }

  @override
  Future<Either<Failure, List<LatLng>>> fetchWayPoints({required List<Waypoint> waypoints}) async => await safeApiCall(() async {
      final PolylinePoints polylinePoints0 = PolylinePoints();
      final response = await _googleApiService.fetchWayPoints(waypoints: waypoints);
      final data = response.data;
      if (data['status'] == 'OK') {
        final String encodedPloyline = data['routes'][0]['overview_polyline']['points'];
        final List<PointLatLng> polylinePoints = polylinePoints0.decodePolyline(encodedPloyline);

        final List<LatLng> polylineCoordinates = polylinePoints.map((e) => LatLng(e.latitude, e.longitude)).toList();
        return polylineCoordinates;
      }
      return [];
    });

  @override
  Future<Either<Failure, List<String>>> fetchDistances(List<String> destinations, LatLng origin) async {
    throw UnimplementedError();
  }

  @override
  Future<Either<Failure, List<PlaceModel>>> searchPlace(String place, LatLng origin) async => safeApiCall(() async {
      final response = await _googleApiService.searchPlace(place);
      final data = response.data;
      if (data['status'] == 'OK') {
        final List<dynamic> unformattedData = data['predictions'];
        final List<Map<String, dynamic>> extractedData = unformattedData
            .map<Map<String, dynamic>>((prediction) => {
          'placeId': prediction['place_id'],
          'title': prediction['structured_formatting']['secondary_text'],
          'subtitle': prediction['description'],
        })
            .toList();
        if (extractedData.isNotEmpty) {
          final placeIds = extractedData.map((e) => e['placeId'] as String).toList();
          final distanceResponse = await _googleApiService.fetchDistances(placeIds, origin);
          final distanceData = distanceResponse.data;

          if (distanceData['status'] == 'OK') {

            final List<PlaceModel> places = extractedData
                .map(
                  (place) => PlaceModel(
                placeId: place['placeId']!,
                title: place['title']!,
                subtitle: place['subtitle']!,
                distance: 'N/A',
                latLng: const LatLng(0.0, 0.0),
              ),
            )
                .toList();
            return places;
          }
        }
      }
      return [];
    });

  @override
  Future<Either<Failure, LatLng>> getLatLngFromPlaceId(String placeId) => safeApiCall(() async {
      final response = await _googleApiService.getLatLngFromPlaceId(placeId);
      final data = response.data;
      if (data['status'] == 'OK') {
        final List<dynamic> results = data['results'];
        if (results.isNotEmpty) {
          final firstResult = results[0];
          final geometry = firstResult['geometry'];
          if (geometry != null) {
            final location = geometry['location'];
            if (location != null) {
              final latitude = location['lat'];
              final longitude = location['lng'];
              if (latitude != null && longitude != null) {
                return LatLng(latitude, longitude);
              }
            }
          }
        }
      }
      return const LatLng(0, 0);
    });
}
